<?php 

namespace KITT3N\Pimcore\RestrictionsBundle\Service;

use Symfony\Component\Security\Core\Security;
use Symfony\Component\HttpFoundation\RequestStack;
use Pimcore\Tool\Admin;
use Pimcore\Tool\Session;
use Pimcore\Model\User as BackendUser;
use Pimcore\Model\User\Role as BackendUserRole;
use Kitt3n\SingleSignOnForSymfonyBundle\Entity\Identity;
use KITT3N\Pimcore\RestrictionsBundle\Model\DataObject\User as FrontendUser;
use Pimcore\Model\DataObject\Group as FrontendGroup;

class SecurityService 
{

    protected $sLoginRoute = 'restrictions_login';

    /**
     * @var Security
     */
    private $security;

    private $requestStack;

    private $configService;

    private $config;

    public function __construct(
        Security $security, 
        ConfigService $configService,
        RequestStack $requestStack
    ) {
        $this->security = $security;
        $this->configService = $configService;
        $this->config = $this->configService->getConfig();
        $this->requestStack = $requestStack;
    }

    /**
     * Return frontend user or null
     * 
     * @return \KITT3N\Pimcore\RestrictionsBundle\Model\DataObject\User|null
     */
    public function getCurrentUser()
    {
        $user = $this->security->getUser();

        return ($user instanceof \KITT3N\Pimcore\RestrictionsBundle\Model\DataObject\User) ? $user : null;
    }

    /**
     * Return backend user or null
     * 
     * @return \Pimcore\Model\User|null
     */
    public function getCurrentBackendUser()
    {
        $user = (
            ($currentBackendUser = Admin::getCurrentUser()) !== null ?
                $currentBackendUser : Session::getReadonly()->get("user")
        );

        return ($user instanceof \Pimcore\Model\User) ? $user : null;
    }

    public function isUserAllowedToDownloadAsset($asset) 
    {
        $currentUser = $this->getCurrentUser();

        $request = $this->requestStack->getCurrentRequest();
        if (!\Pimcore\Tool::isFrontend() || \Pimcore\Tool::isFrontendRequestByAdmin($request)) {
            if($request->query->get('pimcore_preview') === null) {
                return true;
            }
        }

        $assetProperties = $asset->getProperties();

        if ($this->config['assets']['download'] === 'allow') {

            if ($currentUser === null) {
                return true;
            }

            $deniedForGroupIds = [];
            foreach($assetProperties as $assetProperty) {
                if ($assetProperty->getType() === 'object' && substr($assetProperty->getName(), 0, strlen('deny_download')) === 'deny_download') {
                    if (($group = $assetProperty->getData()) instanceof \Pimcore\Model\DataObject\Group) {
                        $deniedForGroupIds[] = $group->getId();
                    }
                }
            }

            if (empty($allowedForGroupIds)) {
                return true;
            }

            $currenUserGroups = $currentUser->getGroups();
            $currentUserGroupIds = [];
            foreach ($currenUserGroups as $group) {
                $currentUserGroupIds[] = $group->getId();
            }

            if (count(array_intersect($currentUserGroupIds, $deniedForGroupIds)) > 0) {
                return false;
            }

            return true;

        }

        if ($this->config['assets']['download'] === 'deny') {

            if ($currentUser === null) {
                return false;
            }

            $allowedForGroupIds = [];
            foreach($assetProperties as $assetProperty) {
                if ($assetProperty->getType() === 'object' && substr($assetProperty->getName(), 0, strlen('allow_download')) === 'allow_download') {
                    if (($group = $assetProperty->getData()) instanceof \Pimcore\Model\DataObject\Group) {
                        $allowedForGroupIds[] = $group->getId();
                    }
                }
            }

            if (empty($allowedForGroupIds)) {
                return false;
            }

            $currenUserGroups = $currentUser->getGroups();
            $currentUserGroupIds = [];
            foreach ($currenUserGroups as $group) {
                $currentUserGroupIds[] = $group->getId();
            }

            if (count(array_intersect($currentUserGroupIds, $allowedForGroupIds)) === 0) {
                return false;
            }

            return true;
        }

        return true;
    }

    
    public function isUserAllowedToAccessDocument($document)
    {

        /* @var array $documentProperties */
        $documentProperties = (($properties = $document->getProperties()) === null ? [] : $properties);

        $currentUser = $this->getCurrentUser();

        $request = $this->requestStack->getCurrentRequest();
        if (!\Pimcore\Tool::isFrontend() || \Pimcore\Tool::isFrontendRequestByAdmin($request)) {
            return true;
        }
        // if ($currentUser instanceof \Pimcore\Bundle\AdminBundle\Security\User\User) {
        //     // Backend User
        //     return true;
        // }

        if ($this->config['documents']['view'] === 'allow') {

            if ($currentUser === null) {
                return true;
            }

            $currenUserGroups = $currentUser->getGroups();
            $currentUserGroupIds = [];
            foreach ($currenUserGroups as $group) {
                $currentUserGroupIds[] = $group->getId();
            }

            $deniedForGroupIds = [];
            foreach($documentProperties as $documentProperty) {
                if ($documentProperty->getType() === 'object' && substr($documentProperty->getName(), 0, strlen('deny')) === 'deny') {
                    if (($group = $documentProperty->getData()) instanceof \Pimcore\Model\DataObject\Group) {
                        $deniedForGroupIds[] = $group->getId();
                    }
                }
            }

            if (count(array_intersect($currentUserGroupIds, $deniedForGroupIds)) > 0) {
                return false;
            }

            return true;

        }

        if ($this->config['documents']['view'] === 'deny') {

            if ($currentUser === null) {
                return false;
            }

            $currenUserGroups = $currentUser->getGroups();
            $currentUserGroupIds = [];
            foreach ($currenUserGroups as $group) {
                $currentUserGroupIds[] = $group->getId();
            }

            $allowedForGroupIds = [];
            foreach($documentProperties as $documentProperty) {
                if ($documentProperty->getType() === 'object' && substr($documentProperty->getName(), 0, strlen('allow')) === 'allow') {
                    if (($group = $documentProperty->getData()) instanceof \Pimcore\Model\DataObject\Group) {
                        $allowedForGroupIds[] = $group->getId();
                    }
                }
            }

            if (count(array_intersect($currentUserGroupIds, $allowedForGroupIds)) === 0) {
                return false;
            }

            return true;
        }

        return true;
    }

    public function getMappedGroupIdentifiers(array $groupIds)
    {
        $groupIdentifiers = [];
        foreach ($groupIds as $groupId) {
            $groupIdentifiers = array_merge($groupIdentifiers, $this->getMappedGroupIdentifier($groupId));
        }

        return $groupIdentifiers;
    }

    public function getMappedGroupIdentifier(string $groupId)
    {
        switch(true) {
            case array_key_exists(($normalizedGroupId = $this->normalizeString($groupId)), $this->config['groups']['mapping']['backend']):
                return $this->config['groups']['mapping']['backend'][$normalizedGroupId];
            case array_key_exists(($normalizedGroupId = $this->normalizeString($groupId)), $this->config['groups']['mapping']['frontend']):
                return $this->config['groups']['mapping']['frontend'][$normalizedGroupId];
            default:
                return [$groupId];
        }
    }

    /**
     * The separator used in keys is typically _ in Yaml and - in XML. For example, auto_connect in Yaml and auto-connect. 
     * The normalization would make both of these auto_connect. 
     * 
     * Normalization is done automatically when getting parameters in the config service as an array.
     * Therefore we need to "normalize" e.g. Azure AD GUIDs before comparing them with our Group mapping keys in yaml config.
     * 
     * @param string $string
     * @return string
     */
    public function normalizeString(string $string)
    { 
        if (strpos($string, '-') !== false && strpos($string, '_') !== false) {
            return $string;
        }
        return $return = str_replace('-', '_', $string);
    }

    /**
     * Create or update backend user from identity
     * 
     * @param Identity $identity
     * @return BackendUser
     */
    public function createOrUpdatePimcoreModelUser(Identity $identity): BackendUser { 
        
        // get backend user
        $pimcoreUser = BackendUser::getByName(
            $identity->getUsername(), 
            [
                'limit' => 1,
                //'active' => true
            ]
        );

        if ( ! $pimcoreUser) {
            // create backend user
            $pimcoreUser = new BackendUser();
        }

        $pimcoreUser->setName($identity->getUsername());
        $pimcoreUser->setPassword($identity->getPassword());
        $pimcoreUser->setEmail($identity->getEmail());
        $pimcoreUser->setFirstname($identity->getFirstname());
        $pimcoreUser->setLastname($identity->getLastname());

        $identityGroupIds = $this->getMappedGroupIdentifiers($identity->getGroupIds());
        $availableUserRoles = new BackendUserRole\Listing();
        $setUserRoles = [];
        foreach ($availableUserRoles as $availableUserRole) {
           if (in_array($availableUserRole->getName(), $identityGroupIds)) {
            $setUserRoles[] = $availableUserRole->getId();
           }
        }
        $pimcoreUser->setRoles($setUserRoles);

        $pimcoreUser->setActive(true);
        $pimcoreUser->setAdmin($identity->getAdmin());
        $pimcoreUser->setParentId($this->config['user']['backend']['parent_id']);
        $pimcoreUser->save();

        return $pimcoreUser;
    }

    /**
     * Create or update a frontend user from identity
     * 
     * @param Identity $identity
     * @param string $fronendGroup
     * @return FrontendUser|\Exception
     */
    public function createOrUpdateRestrictionsBundleModelDataObjectUser(Identity $identity, string $frontendGroup): FrontendUser|\Exception 
    {
        $frontendUserGroupKey = $frontendGroup;

        // The separator used in keys is typically _ in Yaml and - in XML. 
        // For example, auto_connect in Yaml and auto-connect in XML. 
        // The normalization would make both of these auto_connect.
        // 
        // As we use - in our keys/mapping (e.g. Azure AD GUIDs) we have to normalize 
        // before comparing the ['oauth']['groups']['frontend'] value with our key.
        if (array_key_exists(($normalizedFrontendGroup = $this->normalizeString($frontendGroup)), $this->config['groups']['mapping']['frontend'])) { 
            $frontendUserGroupKey = $this->config['groups']['mapping']['frontend'][$normalizedFrontendGroup][0];
        }

        $frontendUserGroup = FrontendGroup::getByKey(
            $frontendUserGroupKey,
            [
                'limit' => 1,
                'unpublished' => false
            ]
        );

        if ( ! $frontendUserGroup) {
            throw new \Exception('Frontend user group not found');
        }
            
        // get frontend user
        $frontendUser = FrontendUser::getByUsername(
            $identity->getUsername(),
            [
                'limit' => 1,
                'unpublished' => true
            ]
        );

        if ( ! $frontendUser) {
            // create backend user
            $frontendUser = new FrontendUser();
        }

        $frontendUser->setKey($identity->getUsername());
        $frontendUser->setUsername($identity->getUsername());
        $frontendUser->setPassword($identity->getPassword());
        $frontendUser->setEmail($identity->getEmail());
        $frontendUser->setFirstname($identity->getFirstname());
        $frontendUser->setLastname($identity->getLastname());
        $frontendUser->setGroup($frontendUserGroup);

        $identityGroupIds = $this->getMappedGroupIdentifiers($identity->getGroupIds());
        $availableGroups = new FrontendGroup\Listing();
        $setGroups = [];
        foreach ($availableGroups as $availableGroup) {
            if (in_array($availableGroup->getKey(), $identityGroupIds)) {
            $setGroups[] = $availableGroup;
            }
        }
        $frontendUser->setGroups($setGroups);

        $frontendUser->setRoles(['ROLE_USER']); // roles are mandatory!
        $frontendUser->setPublished(true);
        $frontendUser->setParentId($this->config['user']['frontend']['parent_id']);
        $frontendUser->save();

        return $frontendUser;
    }

}