<?php 

namespace KITT3N\Pimcore\RestrictionsBundle\Service;

use Symfony\Component\Security\Core\Security;
use Symfony\Component\HttpFoundation\RequestStack;
use Pimcore\Tool\Admin;
use Pimcore\Tool\Session;
use Pimcore\Model\User as BackendUser;
use Pimcore\Model\User\Role as BackendUserRole;
use Kitt3n\SingleSignOnForSymfonyBundle\Entity\Identity;
use KITT3N\Pimcore\RestrictionsBundle\Model\DataObject\User as FrontendUser;
use Pimcore\Model\DataObject\Group as FrontendGroup;

use Symfony\Component\Routing\RouterInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

class SecurityService 
{

    protected $loginRoute = 'restrictions_login';
    protected $forbiddenRoute = 'restrictions_forbidden';

    /**
     * @var Security
     */
    private $security;

    private $requestStack;

    private $configService;

    private $config;

    private $router;

    public function __construct(
        Security $security, 
        ConfigService $configService,
        RequestStack $requestStack,
        RouterInterface $router
    ) {
        $this->security = $security;
        $this->configService = $configService;
        $this->config = $this->configService->getConfig();
        $this->requestStack = $requestStack;
        $this->router = $router;
    }

    /**
     * Return frontend user or null
     * 
     * @return FrontendUser|null
     */
    public function getCurrentUser()
    {
        $user = $this->security->getUser();

        return ($user instanceof FrontendUser) ? $user : null;
    }

    /**
     * Return backend user or null
     * 
     * @return BackendUser|null
     */
    public function getCurrentBackendUser()
    {
        $user = (
            ($currentBackendUser = Admin::getCurrentUser()) !== null ?
                $currentBackendUser : Session::getReadonly()->get("user")
        );

        return ($user instanceof BackendUser) ? $user : null;
    }

    public function isUserAllowedToDownloadAsset($asset) 
    {
        
        $request = $this->requestStack->getCurrentRequest();
        if (!\Pimcore\Tool::isFrontend() || \Pimcore\Tool::isFrontendRequestByAdmin($request)) {
            if($request->query->get('pimcore_preview') === null) {
                return true;
            }
        }

        $currentUser = $this->getCurrentUser();

        $assetProperties = $asset->getProperties();

        if ($this->config['assets']['download'] === 'allow' || (isset($assetProperties['assets.download']) && $assetProperties['assets.download']->getData() === 'allow')) {

            if ($currentUser === null) {
                return true;
            }

            $deniedForGroupIds = [];
            foreach($assetProperties as $assetProperty) {
                if ($assetProperty->getType() === 'object' && substr($assetProperty->getName(), 0, strlen('deny_download')) === 'deny_download') {
                    if (($group = $assetProperty->getData()) instanceof FrontendGroup) {
                        $deniedForGroupIds[] = $group->getId();
                    }
                }
            }

            if (empty($allowedForGroupIds)) {
                return true;
            }

            $currenUserGroups = $currentUser->getGroups();
            $currentUserGroupIds = [];
            foreach ($currenUserGroups as $group) {
                $currentUserGroupIds[] = $group->getId();
            }

            if (count(array_intersect($currentUserGroupIds, $deniedForGroupIds)) > 0) {
                return false;
            }

            return true;

        }

        if ($this->config['assets']['download'] === 'deny' || (isset($assetProperties['assets.download']) && $assetProperties['assets.download']->getData() === 'allow')) {

            if ($currentUser === null) {
                return false;
            }

            $allowedForGroupIds = [];
            foreach($assetProperties as $assetProperty) {
                if ($assetProperty->getType() === 'object' && substr($assetProperty->getName(), 0, strlen('allow_download')) === 'allow_download') {
                    if (($group = $assetProperty->getData()) instanceof FrontendGroup) {
                        $allowedForGroupIds[] = $group->getId();
                    }
                }
            }

            if (empty($allowedForGroupIds)) {
                return false;
            }

            $currenUserGroups = $currentUser->getGroups();
            $currentUserGroupIds = [];
            foreach ($currenUserGroups as $group) {
                $currentUserGroupIds[] = $group->getId();
            }

            if (count(array_intersect($currentUserGroupIds, $allowedForGroupIds)) === 0) {
                return false;
            }

            return true;
        }

        return true;
    }

    /**
     * returns true if a user is allowed to access the given document otherwise false.
     * 
     * @param \Pimcore\Model\Document $document
     * @return bool
     */
    public function isUserAllowedToAccessDocument(\Pimcore\Model\Document $document)
    {

        $request = $this->requestStack->getCurrentRequest();
        if (!\Pimcore\Tool::isFrontend() || \Pimcore\Tool::isFrontendRequestByAdmin($request)) {
            return true;
        }
        // if ($currentUser instanceof \Pimcore\Bundle\AdminBundle\Security\User\User) {
        //     // Backend User
        //     return true;
        // }

        /* @var array $documentProperties */
        $documentProperties = (($properties = $document->getProperties()) === null ? [] : $properties);

        $currentUser = $this->getCurrentUser();

        if ($this->config['documents']['view'] === 'allow' || (isset($documentProperties['documents.view']) && $documentProperties['documents.view']->getData() === 'allow')) {

            if ($currentUser === null) {
                return true;
            }

            $currenUserGroups = $currentUser->getGroups();
            $currentUserGroupIds = [];
            foreach ($currenUserGroups as $group) {
                $currentUserGroupIds[] = $group->getId();
            }

            $deniedForGroupIds = [];
            foreach($documentProperties as $documentProperty) {
                if ($documentProperty->getType() === 'object' && substr($documentProperty->getName(), 0, strlen('deny')) === 'deny') {
                    if (($group = $documentProperty->getData()) instanceof FrontendGroup) {
                        $deniedForGroupIds[] = $group->getId();
                    }
                }
            }

            if (count(array_intersect($currentUserGroupIds, $deniedForGroupIds)) > 0) {
                return false;
            }

            return true;

        }

        if ($this->config['documents']['view'] === 'deny' || (isset($documentProperties['documents.view']) && $documentProperties['documents.view']->getData() === 'deny')) {

            if ($currentUser === null) {
                return false;
            }

            $currenUserGroups = $currentUser->getGroups();
            $currentUserGroupIds = [];
            foreach ($currenUserGroups as $group) {
                $currentUserGroupIds[] = $group->getId();
            }

            $allowedForGroupIds = [];
            foreach($documentProperties as $documentProperty) {
                if ($documentProperty->getType() === 'object' && substr($documentProperty->getName(), 0, strlen('allow')) === 'allow') {
                    if (($group = $documentProperty->getData()) instanceof FrontendGroup) {
                        $allowedForGroupIds[] = $group->getId();
                    }
                }
            }

            if (count(array_intersect($currentUserGroupIds, $allowedForGroupIds)) === 0) {
                return false;
            }

            return true;
        }

        return true;
    }

    /**
     * Returns true if a user is allowed to access the given object otherwise false.
     * 
     * @param \Pimcore\Model\DataObject\Concrete $object
     * @return bool
     */
    public function isUserAllowedToAccessObject(\Pimcore\Model\DataObject\Concrete $object)
    {

        $request = $this->requestStack->getCurrentRequest();
        if (!\Pimcore\Tool::isFrontend() || \Pimcore\Tool::isFrontendRequestByAdmin($request)) {
            return true;
        }
        // if ($currentUser instanceof \Pimcore\Bundle\AdminBundle\Security\User\User) {
        //     // Backend User
        //     return true;
        // }

        /* @var array $documentProperties */
        $objectProperties = (($properties = $object->getProperties()) === null ? [] : $properties);

        $currentUser = $this->getCurrentUser();

        if ($this->config['objects']['view'] === 'allow' || (isset($objectProperties['objects.view']) && $objectProperties['objects.view']->getData() === 'allow')) {

            if ($currentUser === null) {
                return true;
            }

            $currenUserGroups = $currentUser->getGroups();
            $currentUserGroupIds = [];
            foreach ($currenUserGroups as $group) {
                $currentUserGroupIds[] = $group->getId();
            }

            $deniedForGroupIds = [];
            foreach($objectProperties as $objectProperty) {
                if ($objectProperty->getType() === 'object' && substr($objectProperty->getName(), 0, strlen('deny')) === 'deny') {
                    if (($group = $objectProperty->getData()) instanceof FrontendGroup) {
                        $deniedForGroupIds[] = $group->getId();
                    }
                }
            }

            if (count(array_intersect($currentUserGroupIds, $deniedForGroupIds)) > 0) {
                return false;
            }

            return true;

        }

        if ($this->config['objects']['view'] === 'deny' || (isset($objectProperties['objects.view']) && $objectProperties['objects.view']->getData() === 'deny')) {

            if ($currentUser === null) {
                return false;
            }

            $currenUserGroups = $currentUser->getGroups();
            $currentUserGroupIds = [];
            foreach ($currenUserGroups as $group) {
                $currentUserGroupIds[] = $group->getId();
            }

            $allowedForGroupIds = [];
            foreach($objectProperties as $objectProperty) {
                if ($objectProperty->getType() === 'object' && substr($objectProperty->getName(), 0, strlen('allow')) === 'allow') {
                    if (($group = $objectProperty->getData()) instanceof FrontendGroup) {
                        $allowedForGroupIds[] = $group->getId();
                    }
                }
            }

            if (count(array_intersect($currentUserGroupIds, $allowedForGroupIds)) === 0) {
                return false;
            }

            return true;
        }

        return true;
    }

    /**
     * Is user allowed to access content based on a given group key?
     * 
     * Replace method accessGranted() in kitt3n/pimcore-members
     * 
     * @param mixed $groupKey
     * @return bool
     */
    public function isUserAllowedToAccessContent($groupKey) {
    
        $request = $this->requestStack->getCurrentRequest();
        if (!\Pimcore\Tool::isFrontend() || \Pimcore\Tool::isFrontendRequestByAdmin($request)) {
            return true;
        }

        $currentUser = $this->getCurrentUser();

        if ($this->config['objects']['view'] === 'allow') {

            if ($currentUser === null) {
                return true;
            }

            $currenUserGroups = $currentUser->getGroups();
            $currentUserGroupKeys = [];
            foreach ($currenUserGroups as $group) {
                $currentUserGroupKeys[] = $group->getKey();
            }

            $deniedForGroupKeys = [
                $groupKey
            ];

            if (count(array_intersect($currentUserGroupKeys, $deniedForGroupKeys)) > 0) {
                return false;
            }

            return true;

        }

        if ($this->config['objects']['view'] === 'deny') {

            if ($currentUser === null) {
                return false;
            }

            $currenUserGroups = $currentUser->getGroups();
            $currentUserGroupKeys = [];
            foreach ($currenUserGroups as $group) {
                $currentUserGroupKeys[] = $group->getKey();
            }

            $allowedForGroupKeys = [
                $groupKey
            ];
        
            if (count(array_intersect($currentUserGroupKeys, $allowedForGroupKeys)) === 0) {
                return false;
            }

            return true;

        }

        return true;

    }

    /**
     * Force redirect to login page if needed.
     * 
     * @return RedirectResponse|false
     */
    public function forceRedirectToLogin(): RedirectResponse|false
    {
        $request = $this->requestStack->getCurrentRequest();
        if (!\Pimcore\Tool::isFrontend() || \Pimcore\Tool::isFrontendRequestByAdmin($request)) {
            return false;
        }

        $currentUser = $this->getCurrentUser();
        if ($currentUser === null) {
            $url = $this->router->generate($this->loginRoute);
            $response = new RedirectResponse($url);
            return $response;
        }

        return false;
    }

    /**
     * Redirect user if necessary. This is used for example if the user is not allowed to access content for a specific group. 
     * Redirect to 403 page or login page.
     * 
     * @param \Pimcore\Model\DataObject\Concrete|\Pimcore\Model\Document $object
     * @return RedirectResponse|false
     */
    public function redirectUserIfNecessaryForGivenGroup(string $groupKey): RedirectResponse|false 
    {

        $request = $this->requestStack->getCurrentRequest();
        if (!\Pimcore\Tool::isFrontend() || \Pimcore\Tool::isFrontendRequestByAdmin($request)) {
            return false;
        }

        $isUserAllowedToAccessGivenObject = $this->isUserAllowedToAccessContent($groupKey);

        if ( ! $isUserAllowedToAccessGivenObject && $this->getCurrentUser() !== null) {
            $url = $this->router->generate($this->forbiddenRoute);
            $response = new RedirectResponse($url);
            return $response;
        }

        if ( ! $isUserAllowedToAccessGivenObject && $this->getCurrentUser() === null) {
            $url = $this->router->generate($this->loginRoute);
            $response = new RedirectResponse($url);
            return $response;
        }

        return false;

    }

    /**
     * Redirect user if necessary. This is used for example if the user is not allowed to access the object or document. 
     * Redirect to 403 page or login page.
     * 
     * @param \Pimcore\Model\DataObject\Concrete|\Pimcore\Model\Document $object
     * @return RedirectResponse|false
     */
    public function redirectUserIfNecessaryForGivenObject(\Pimcore\Model\DataObject\Concrete|\Pimcore\Model\Document $object): RedirectResponse|false
    {

        $request = $this->requestStack->getCurrentRequest();
        if (!\Pimcore\Tool::isFrontend() || \Pimcore\Tool::isFrontendRequestByAdmin($request)) {
            return false;
        }
        
        if ($object instanceof \Pimcore\Model\DataObject\Concrete) {
            $isUserAllowedToAccessGivenObject = $this->isUserAllowedToAccessObject($object);
        }

        if ($object instanceof \Pimcore\Model\Document) {
            $isUserAllowedToAccessGivenObject = $this->isUserAllowedToAccessDocument($object);
        }

        if ( ! $isUserAllowedToAccessGivenObject && $this->getCurrentUser() !== null) {
            $url = $this->router->generate($this->forbiddenRoute);
            $response = new RedirectResponse($url);
            return $response;
        }

        if ( ! $isUserAllowedToAccessGivenObject && $this->getCurrentUser() === null) {
            $url = $this->router->generate($this->loginRoute);
            $response = new RedirectResponse($url);
            return $response;
        }

        return false;

    }

    public function getMappedGroupIdentifiers(array $groupIds)
    {
        $groupIdentifiers = [];
        foreach ($groupIds as $groupId) {
            $groupIdentifiers = array_merge($groupIdentifiers, $this->getMappedGroupIdentifier($groupId));
        }

        return $groupIdentifiers;
    }

    public function getMappedGroupIdentifier(string $groupId)
    {
        switch(true) {
            case array_key_exists(($normalizedGroupId = $this->normalizeString($groupId)), $this->config['groups']['mapping']['backend']) 
            && array_key_exists(($normalizedGroupId = $this->normalizeString($groupId)), $this->config['groups']['mapping']['frontend']):
                return array_merge(
                    $this->config['groups']['mapping']['backend'][$normalizedGroupId], 
                    $this->config['groups']['mapping']['frontend'][$normalizedGroupId]
                );
            case array_key_exists(($normalizedGroupId = $this->normalizeString($groupId)), $this->config['groups']['mapping']['backend']):
                return $this->config['groups']['mapping']['backend'][$normalizedGroupId];
            case array_key_exists(($normalizedGroupId = $this->normalizeString($groupId)), $this->config['groups']['mapping']['frontend']):
                return $this->config['groups']['mapping']['frontend'][$normalizedGroupId];
            default:
                return [$groupId];
        }
    }

    /**
     * The separator used in keys is typically _ in Yaml and - in XML. For example, auto_connect in Yaml and auto-connect. 
     * The normalization would make both of these auto_connect. 
     * 
     * Normalization is done automatically when getting parameters in the config service as an array.
     * Therefore we need to "normalize" e.g. Azure AD GUIDs before comparing them with our Group mapping keys in yaml config.
     * 
     * @param string $string
     * @return string
     */
    public function normalizeString(string $string)
    { 
        if (strpos($string, '-') !== false && strpos($string, '_') !== false) {
            return $string;
        }
        return $return = str_replace('-', '_', $string);
    }

    /**
     * Create or update backend user from identity
     * 
     * @param Identity $identity
     * @return BackendUser
     */
    public function createOrUpdatePimcoreModelUser(Identity $identity): BackendUser { 
        
        // get backend user
        $pimcoreUser = BackendUser::getByName(
            $identity->getUsername(), 
            [
                'limit' => 1,
                //'active' => true
            ]
        );

        if ( ! $pimcoreUser) {
            // create backend user
            $pimcoreUser = new BackendUser();
        }

        $pimcoreUser->setName($identity->getUsername());
        $pimcoreUser->setPassword($identity->getPassword());
        $pimcoreUser->setEmail($identity->getEmail());
        $pimcoreUser->setFirstname($identity->getFirstname());
        $pimcoreUser->setLastname($identity->getLastname());

        $identityGroupIds = $this->getMappedGroupIdentifiers($identity->getGroupIds());
        $availableUserRoles = new BackendUserRole\Listing();
        $setUserRoles = [];
        foreach ($availableUserRoles as $availableUserRole) {
           if (in_array($availableUserRole->getName(), $identityGroupIds)) {
            $setUserRoles[] = $availableUserRole->getId();
           }
        }
        $pimcoreUser->setRoles($setUserRoles);

        $pimcoreUser->setActive(true);
        $pimcoreUser->setAdmin($identity->getAdmin());
        $pimcoreUser->setParentId($this->config['user']['backend']['parent_id']);
        $pimcoreUser->save();

        return $pimcoreUser;
    }

    /**
     * Create or update a frontend user from identity
     * 
     * @param Identity $identity
     * @param string $fronendGroup
     * @return FrontendUser|\Exception
     */
    public function createOrUpdateRestrictionsBundleModelDataObjectUser(Identity $identity, string $frontendGroup): FrontendUser|\Exception 
    {
        $frontendUserGroupKey = $frontendGroup;

        // The separator used in keys is typically _ in Yaml and - in XML. 
        // For example, auto_connect in Yaml and auto-connect in XML. 
        // The normalization would make both of these auto_connect.
        // 
        // As we use - in our keys/mapping (e.g. Azure AD GUIDs) we have to normalize 
        // before comparing the ['oauth']['groups']['frontend'] value with our key.
        if (array_key_exists(($normalizedFrontendGroup = $this->normalizeString($frontendGroup)), $this->config['groups']['mapping']['frontend'])) { 
            $frontendUserGroupKey = $this->config['groups']['mapping']['frontend'][$normalizedFrontendGroup][0];
        }

        $frontendUserGroup = FrontendGroup::getByKey(
            $frontendUserGroupKey,
            [
                'limit' => 1,
                'unpublished' => false
            ]
        );

        if ( ! $frontendUserGroup) {
            throw new \Exception("Frontend user group '$frontendUserGroupKey' from config.yaml not found in system");
        }
            
        // get frontend user
        $frontendUser = FrontendUser::getByUsername(
            $identity->getUsername(),
            [
                'limit' => 1,
                'unpublished' => true
            ]
        );

        if ( ! $frontendUser) {
            // create backend user
            $frontendUser = new FrontendUser();
        }

        $frontendUser->setKey($identity->getUsername());
        $frontendUser->setUsername($identity->getUsername());
        $frontendUser->setPassword($identity->getPassword());
        $frontendUser->setEmail($identity->getEmail());
        $frontendUser->setFirstname($identity->getFirstname());
        $frontendUser->setLastname($identity->getLastname());
        $frontendUser->setGroup($frontendUserGroup);

        $identityGroupIds = $this->getMappedGroupIdentifiers($identity->getGroupIds());
        $availableGroups = new FrontendGroup\Listing();
        $setGroups = [];

        // It is possible to create a user e.g.in Azure with only the backend user group. 
        // In Pimcore this makes not that much sense because the user would not be able to view document changes 
        // in the frontend.
        // 
        // Therefore we add the frontend group whether the e.g. Azure user has access to the frontend or not.
        if ( ! $identity->getFrontendAccess() and $identity->getBackendAccess()) { 
            $setGroups[] = $frontendUserGroup;
        }
        
        foreach ($availableGroups as $availableGroup) {
            if($availableGroup == $frontendUserGroup and in_array($availableGroup, $setGroups)) { 
                // $frontendUserGroup is already in $setGroups array, we don't want to add it again.
                continue; 
            }
            if (in_array($availableGroup->getKey(), $identityGroupIds)) {
            $setGroups[] = $availableGroup;
            }
        }
        $frontendUser->setGroups($setGroups);

        $frontendUser->setRoles(['ROLE_USER']); // roles are mandatory!
        $frontendUser->setPublished(true);
        $frontendUser->setParentId($this->config['user']['frontend']['parent_id']);
        $frontendUser->save();

        return $frontendUser;
    }

    public function getContext() 
    {
        $request = $this->requestStack->getCurrentRequest();
        if (!\Pimcore\Tool::isFrontend() || \Pimcore\Tool::isFrontendRequestByAdmin($request)) {
            return 'backend';
        }
        return 'frontend';
    }

}