<?php 

namespace Kitt3n\SingleSignOnForSymfonyBundle\Controller;

use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\RequestStack;

use Kitt3n\SingleSignOnForSymfonyBundle\Service\CryptoService;
use Kitt3n\SingleSignOnForSymfonyBundle\Service\CookieService;
use Kitt3n\SingleSignOnForSymfonyBundle\Service\ConfigService;
use Kitt3n\SingleSignOnForSymfonyBundle\Service\OauthService;
use League\OAuth2\Client\Provider\GenericProvider;
use Kitt3n\SingleSignOnForSymfonyBundle\Entity\Identity;
use Symfony\Component\HttpFoundation\Request;
use League\OAuth2\Client\Token\AccessTokenInterface;


class SingleSignOnController extends AbstractController
{
    protected $cookieName = 'ssofs';

    protected $sessionName = 'ssofs';

    protected $tokenProvider = 'generic';

    protected $requestStack;

    protected $oauthService;

    protected $configService;

    protected $cookieService;

    protected $routeToConnectAction = 'single_sign_on_for_symfony_generic_connect';

    protected $routeToRefreshTokenAction = 'single_sign_on_for_symfony_generic_refresh_token';

    protected $errorTemplate = '@SingleSignOnForSymfonyBundle/single-sign-on/error.html.twig';

    protected $successTemplate = '@SingleSignOnForSymfonyBundle/single-sign-on/success.html.twig';

    /** @var GenericProvider $oauthClient */
    protected $oauthClient;

    protected $config;

    public function __construct(
        RequestStack $requestStack,
        OauthService $oauthService,
        ConfigService $configService,
        CookieService $cookieService,
    ) {
        $this->requestStack = $requestStack;

        $this->configService = $configService;
        $this->config = $configService->getConfig();

        $this->oauthService = $oauthService;
        $this->oauthClient = $oauthService->getOauthClient(
            $this->config['oauth']['client_id'],
            $this->config['oauth']['client_secret'],
            $this->config['oauth']['tenant'],
            $this->config['oauth']['redirect_uri'],
            $this->config['oauth']['url_authorize'],
            $this->config['oauth']['url_access_token'],
            $this->config['oauth']['url_resource_owner_details'],
            $this->config['oauth']['scopes']
        );
        
        $this->cookieService = $cookieService;
    }

    /**
     * @Route("/{_locale}/sso/generic/connect", name="single_sign_on_for_symfony_generic_connect"),
     *          defaults={
     *         "_locale": "en"
     *     }
     */
    public function connectAction(Request $request): Response
    {
        
        $session = $request->getSession();

        $accessToken = $session->get('SingleSignOnController::processAction::accessToken');

        if ($accessToken instanceof AccessTokenInterface) {
            return $this->redirectToRoute($this->routeToRefreshTokenAction);
        }

        $authUrl = $this->oauthClient->getAuthorizationUrl();

        $oauthState = $this->oauthClient->getState();
        $session->set('oauthState', $oauthState);

        return new RedirectResponse($authUrl);

    }

    /**
     * @Route("/{_locale}/sso/generic/refresh-token", name="single_sign_on_for_symfony_generic_refresh_token"),
     *          defaults={
     *         "_locale": "en"
     *     }
     */
    public function refreshTokenAction(Request $request): Response
    {

        $session = $request->getSession();

        $currentAccessToken = $session->get('SingleSignOnController::processAction::accessToken');

        if (null === $currentAccessToken) {
            return $this->redirectToRoute($this->routeToConnectAction);
        }

        try {

            $accessToken = $this->oauthClient->getAccessToken(
                'refresh_token', 
                [
                    'refresh_token' => $currentAccessToken->getRefreshToken(),
                ]
            );
    
            $session->set('SingleSignOnController::processAction::accessToken', $accessToken);
            $session->set('SingleSignOnController::processAction::accessTokenProvider', $this->tokenProvider);

        } catch (\League\OAuth2\Client\Provider\Exception\IdentityProviderException $e) {

            if($session->get('SingleSignOnController::processAction::redirect') === 1) {
                $parameters['error'] = $e->getMessage() . '.';
                $parameters['suggestion'] = 'Please try again.';
                $session->set('SingleSignOnController::processAction::redirect', null);
                $session->set('SingleSignOnController::processAction::state', 'error');
                return $this->render($this->errorTemplate, $parameters);
            }

            $session->set('SingleSignOnController::processAction::redirect', 1);
            return $this->redirectToRoute($this->routeToConnectAction);

        } catch (\Exception $e) {
            $session->set('SingleSignOnController::processAction::accessToken', null);
            $session->set('SingleSignOnController::processAction::accessTokenProvider', $this->tokenProvider);
            $session->set('SingleSignOnController::processAction::redirect', 1);
            return $this->redirectToRoute($this->routeToConnectAction);
        }

        if($accessToken->hasExpired()) {
            
            if($session->get('SingleSignOnController::processAction::redirect') === 1) {
                $parameters['error'] = 'Access token has expired.';
                $parameters['suggestion'] = 'Please login again.';
                $session->set('SingleSignOnController::processAction::redirect', null);
                $session->set('SingleSignOnController::processAction::state', 'error');
                return $this->render($this->errorTemplate, $parameters);
            }

            $session->set('SingleSignOnController::processAction::redirect', 1);
            return $this->redirectToRoute($this->routeToConnectAction);

        }

        $identity = $this->createIdentity($accessToken);

        $this->persistIdentity($identity);

        return $this->render($this->successTemplate, []);

    }

    /**
     * @Route("/{_locale}/sso/generic/process", name="single_sign_on_for_symfony_process"),
     *          defaults={
     *         "_locale": "en"
     *     }
     */
    public function processAction(Request $request): Response
    {
        /**
         * Get error from query parameters
         */
        $oauthErrorFromQuery = $request->query->get('error');

        if(isset($oauthErrorFromQuery)) { 
            $oauthErrorUriFromQuery = $request->query->get('error_uri');
            if(isset($oauthErrorUriFromQuery)) { 
                return $this->redirect(
                    $oauthErrorUriFromQuery,
                    302
                );
            }
        }

        $session = $request->getSession();

        $parameters = [];

        /**
         * Get OAuth state from session set in ssoAzureAdConnectAction
         * and remove it from session
         */
        $oauthStateFromSession = $session->get('oauthState');
        $session->set('oauthState', null);

        /**
         * Get state from query parameters
         */
        $oauthStateFromQuery = $request->query->get('state');
       
         /**
         * OAuth state may not be present in the session if 
         * for example /sso/<generic?/process?... is called accidentally a second time
         */
        if (!isset($oauthStateFromSession)) {
            
            if($session->get('SingleSignOnController::processAction::redirect') === 1) {
                $parameters['error'] = 'OAuth state not set.';
                $parameters['suggestion'] = 'Please try again.';
                $session->set('SingleSignOnController::processAction::redirect', null);
                $session->set('SingleSignOnController::processAction::state', 'error');
                return $this->render($this->errorTemplate, $parameters);
            }

            $session->set('SingleSignOnController::processAction::redirect', 1);
            return $this->redirectToRoute($this->routeToConnectAction);
            
        }

        if (!isset($oauthStateFromQuery)) {

            if($session->get('SingleSignOnController::processAction::redirect') === 1) {
                $parameters['error'] = 'Provided state is not set.';
                $parameters['suggestion'] = 'Please try again.';
                $session->set('SingleSignOnController::processAction::redirect', null);
                $session->set('SingleSignOnController::processAction::state', 'error');
                return $this->render($this->errorTemplate, $parameters);
            }

            $session->set('SingleSignOnController::processAction::redirect', 1);
            return $this->redirectToRoute($this->routeToConnectAction);

        }

        if ($oauthStateFromSession != $oauthStateFromQuery) {

            if($session->get('SingleSignOnController::processAction::redirect') === 1) {
                $parameters['error'] = 'Provided state does not match expected state.';
                $parameters['suggestion'] = 'Please try again.';
                $session->set('SingleSignOnController::processAction::redirect', null);
                $session->set('SingleSignOnController::processAction::state', 'error');
                return $this->render($this->errorTemplate, $parameters);
            }

            $session->set('SingleSignOnController::processAction::redirect', 1);
            return $this->redirectToRoute($this->routeToConnectAction);

        }

        $authCode = $request->query->get('code');
        if (!isset($authCode)) {

            if($session->get('SingleSignOnController::processAction::redirect') === 1) {
                $parameters['error'] = 'OAuth code is not set.';
                $parameters['suggestion'] = 'Please try again.';
                $session->set('SingleSignOnController::processAction::redirect', null);
                $session->set('SingleSignOnController::processAction::state', 'error');
                return $this->render($this->errorTemplate, $parameters);
            }

            $session->set('SingleSignOnController::processAction::redirect', 1);
            return $this->redirectToRoute($this->routeToConnectAction);

        }

        try {

            $session->set('SingleSignOnController::processAction::accessToken', null);
            $session->set('SingleSignOnController::processAction::accessTokenProvider', $this->tokenProvider);

            $accessToken = $this->oauthClient->getAccessToken(
                'authorization_code', 
                [
                    'code' => $authCode
                ]
            );

            $session->set('SingleSignOnController::processAction::accessToken', $accessToken);
            $session->set('SingleSignOnController::processAction::accessTokenProvider', $this->tokenProvider);

        } catch (\League\OAuth2\Client\Provider\Exception\IdentityProviderException $e) {

            if($session->get('SingleSignOnController::processAction::redirect') === 1) {
                $parameters['error'] = $e->getMessage() . '.';
                $parameters['suggestion'] = 'Please try again.';
                $session->set('SingleSignOnController::processAction::redirect', null);
                $session->set('SingleSignOnController::processAction::state', 'error');
                return $this->render($this->errorTemplate, $parameters);
            }

            $session->set('SingleSignOnController::processAction::redirect', 1);
            return $this->redirectToRoute($this->routeToConnectAction);

        }

        if($accessToken->hasExpired()) {
            
            if($session->get('SingleSignOnController::processAction::redirect') === 1) {
                $parameters['error'] = 'Access token has expired.';
                $parameters['suggestion'] = 'Please login again.';
                $session->set('SingleSignOnController::processAction::redirect', null);
                $session->set('SingleSignOnController::processAction::state', 'error');
                return $this->render($this->errorTemplate, $parameters);
            }

            $session->set('SingleSignOnController::processAction::redirect', 1);
            return $this->redirectToRoute($this->routeToConnectAction);

        }

        $identity = $this->createIdentity($accessToken);

        $this->persistIdentity($identity);

        $session->set('SingleSignOnController::processAction::redirect', null);
        $session->set('SingleSignOnController::processAction::state', 'success');

        return $this->render($this->successTemplate, $parameters);
       
    }

    protected function createIdentity(AccessTokenInterface $accessToken)
    { 
        return $identity = new Identity(
            token: $token = $accessToken->getToken(),
            time: time(),
            expires: $expires = $accessToken->getExpires(),
        );
    }

    public function persistIdentity(Identity $identity)
    { 
        $serializedIdentity = serialize($identity);

        if($this->config['handlers']['session']) {

            $request = $this->requestStack->getCurrentRequest();
            $session = $request->getSession();

            $encryptedSerializedIdentity = CryptoService::encrypt(
                message: $serializedIdentity,
                key: $this->config['encryption']['secret']
            );

            $session->set($this->sessionName, $encryptedSerializedIdentity);

        }

        if($this->config['handlers']['cookie']) {
            CookieService::saveLargeCookie(
                name: $this->cookieName,
                value: $serializedIdentity,
                expires: $identity->getExpires(),
                path: '/',
                encrypt: true,
                key: $this->config['encryption']['secret'],
            );
        }
    }
	
}	
